#-------------------------------------------------------------------------------
# Name:        climate_change  version 2.0
# Purpose:     module for dynamic climate change impact assessment
#
# version 1
# Author:      Shimako
# Models and data based on IPCC AR4
#
# Version 2
#
# Modified: Bayomie 03/06/2024 Convolution calculation replaced by the Python dedicated function (eliminates loops)
# Modified: Bayomie 17/07/2024 Adding methane oxidation section in convolution_cc
# Modified: Barna 17/07/2024
# - carbon-climate feedback added on radiative forcing by the function RFDindirect
# - methane oxidation corrected for correct Carbon balance and time horizon 
# - methane RF and T includes naw the contribution of the neo-formed CO2
# - Hydrogen was added in the Data files (Hauglustain, 2022)
#
# 
# It needs to be adapted to the DyPLCA output template in order to be compatible with DyPLCA 
#
#
#
# !!!! remeber to input CO2 soil
#-------------------------------------------------------------------------------
import numpy as np
import csv
import os
import gc
from numpy import array, identity, dot, matrix, zeros, linalg
from math import exp
from sys import getsizeof
from datetime import date
import matplotlib.pyplot as plt

from generic_functions import write_csv, reduce_data, write_report, \
                              create_graphic

from substances_classes import Cc_substances,Dyplca_substances

class Climate_change(object):
    """
    Class for climate change impact

    parameters
    ----------
        substances_dlci (list): list with substances of dlci
        time_dlci (list): time vector of dlci in years
        span_time_cc (float): span time for cc calculation
        discretization_interval_cc (float) : step for convolution
        path (dict) : paths for saving
        report (list of lists) : give final report for cc
        cc_substances (list) : list of substances for cc method
        time_dlci_norm (list) : normalized vector of dlci time
        time_cc (list) : time vector for cc calculation (w/out normalization)
        time_cc_calculation (list) : time vector for cc calc (normalized)
        global_gwp_result (float) = conventional results for gwp
        global_gtp_result (float)=conventional results for gtp
        aggregated_gwp_result (float)= aggregated results for dynamic rad forcing
        aggregated_gtp_result (float)= aggregated results for dynamic temperature
        step_dlci (float)= time step size of dlci
        step_calculation (float)= time step size of climate change calculation
    """

    def __init__(self, project, time_horizon_cc, discretization_interval_cc, cc_aggregation_dlci_step):

        new_int = project.aggregate_values(cc_aggregation_dlci_step)

        self.substances_dlci = project.substances
        self.time_dlci = project.time
        self.span_time_cc = time_horizon_cc
        self.discretization_interval_cc = discretization_interval_cc
        self.path = project.path
        self.report = [['Project ' + project.name + ' created in '+ str(date.today())], \
                      [' '], \
                      ['Climate Change report'], \
                      [' '], \
                      ['Values for Climate Change'], \
                      ['substance_id','substance','air subcompartment', \
                       'dyn temperature (K)','dyn radiative (W.m-2yr']]
        self.cc_substances = []
        self.time_dlci_norm = []
        self.time_cc = []
        self.time_cc_calculation = []
        self.global_gwp_result = []
        self.global_gtp_result =[]
        self.cumulated_rf = []
        self.cumulated_temp = []
        self.aggregated_gwp_result = []
        self.aggregated_gtp_result = []
        self.step_dlci = 0.0
        self.step_calculation = 0.0

        self.import_cc_files()
        
        #print(project.time[:])

    def climate_change_main(self):
        """
        Calculation of climate change

        Call the main methods for the calculation of the method
        """
        print('Climate Change calculation')
        self.time_vector_handling() # time vector handling
        self.calculation_material_balance()
        write_report ('cc_report', self.report, self.path["cc"])
        
        

    def time_vector_handling(self):
        """
        Adaptation of time vectors

        Adapts and creates the vectors of time for the dynamic impact assessment.
        Converts in days, normalize for the calculation (avoid negative values
        in the time vector)
        """
       

        self.time_dlci = [float(i) for i in self.time_dlci]
        
        print('first time point') 
        print(self.time_dlci[0])
        
        self.time_cc = np.arange(self.time_dlci[0],
                                 self.span_time_cc,
                                 self.discretization_interval_cc)
        
        initial_time = self.time_dlci[0] # value for normalization
   
        self.time_dlci_norm = [a-initial_time for a in self.time_dlci] # vector that comes from DyPLCA
        self.time_cc_calculation = [a-initial_time for a in self.time_cc] # vector for calculation ( horizon time)
        self.step_dlci = (self.time_dlci[-1]-self.time_dlci[0])/(len(self.time_dlci)-1)
        self.step_calculation = self.discretization_interval_cc

    
    def calculation_material_balance(self):
        """
        Calculation of material balance and impacts

        Searches substances in the DyPLCA inventory file for the dynamic
        climate change assessment and perform the calculations. It also includes a
        function for the calculation of the methane oxidation
        """
        print('Methane handling')
        self.methane_oxidation()
        print('continuation...')
        # ______________________________________________________________________________
        # calculation of material balance
        final_dyn_temperature = []
        final_dyn_radiative = []
        substances_evaluated_cc = []
        for s in range(len(self.substances_dlci)):
            print(str(s+1) + '/' + str(len(self.substances_dlci)))
            for i in range(len(self.cc_substances)):

                if (self.substances_dlci[s].name == self.cc_substances[i].name and
                    self.substances_dlci[s].compartment == self.cc_substances[i].compartment):

                    print("self.substances_dlci[s].name")
                    print(self.substances_dlci[s].name)

                    # ____________________
                    # needs to be in years because of units
                    # calculation of convolution
                    if sum(self.substances_dlci[s].emission)== 0.0:
                        continue
                    results_convolution, dyn_temperature, dyn_radiative_forcing = self.convolution_cc(self.substances_dlci[s],self.cc_substances[i])
                    self.write_results(self.substances_dlci[s].name,
                                  self.substances_dlci[s].compartment,
                                  self.substances_dlci[s].subcompartment,
                                  dyn_temperature,
                                  'temperature',
                                  self.path['cc'])
                    self.write_results(self.substances_dlci[s].name,
                                  self.substances_dlci[s].compartment,
                                  self.substances_dlci[s].subcompartment,
                                  dyn_radiative_forcing,
                                  'radiative',
                                  self.path['cc'])
                    int_dyn_r  = self.impacts_integral(dyn_radiative_forcing,
                                                       self.time_cc_calculation)

                    # ----------------------------------------------------------
                    # calculation of results for all substances
                    # ----------------------------------------------------------
                    value_dynt =  dyn_temperature[-1]
                    temp_is_nan = np.isnan(sum(dyn_temperature))
                    if self.global_gtp_result == []:
                        self.global_gtp_result = [0.0]*len(dyn_temperature)
                    if (sum(dyn_temperature)!= 0.0 and not temp_is_nan):
#                        for jj in range(len(dyn_temperature)):
                        self.global_gtp_result = [a+b for a,b in zip(self.global_gtp_result,dyn_temperature)]
                    if self.global_gwp_result == []:
                        self.global_gwp_result = [0.0]*len(dyn_radiative_forcing)
                    if (sum(dyn_radiative_forcing)!= 0.0 and not temp_is_nan):
                        self.global_gwp_result = [a+b for a,b in zip(self.global_gwp_result,dyn_radiative_forcing)]
                    # ----------------------------------------------------------
                    # report updating and temperature graphic creation
                    # ----------------------------------------------------------
                    self.graphs_cc(dyn_temperature,
                                   self.substances_dlci[s].name,
                                   self.substances_dlci[s].subcompartment)
                    int_mass = sum(self.substances_dlci[s].emission)*self.step_dlci
                    self.report.append([i,self.substances_dlci[s].name,
                                        self.substances_dlci[s].subcompartment,
                                        value_dynt,
                                        int_dyn_r,
                                        int_mass]
                                        )
                    break
        #-----------------------------------------------------------------------
        # calculation of cumulated vector and cumulated results for temp and rf
        #-----------------------------------------------------------------------
        self.cumulated_calculation ()
        total_radiative = self.impacts_integral (self.global_gwp_result,
                                                 self.time_cc_calculation
                                                 )
        total_temp = self.impacts_integral (self.global_gtp_result,
                                                 self.time_cc_calculation
                                                 )
        #-------------------------------------------------------
        # write final climate change report
        #-------------------------------------------------------
        if len(self.global_gtp_result)!=0: # check if there is results, if yes, do
#            print "global_gtp_result"
#            print self.global_gtp_result
            self.report.append(["",""])
            self.report.append(["Final Temperature (K)","Total Radiative (W.m-2.yr)"])
            self.report.append([self.global_gtp_result[-1],total_radiative])
            # write total results
            self.write_final_results(self.path["cc"])
            self.final_graphs_cc(self.global_gwp_result, self.cumulated_rf,
                                 self.global_gtp_result, self.cumulated_temp)

    def convolution_cc(self, substance_dlci, substance_cc):
        """
        Convolution of a substance in the climate change model.
    
        Performs the convolution calculation and the impacts calculation for a
        given substance, including the handling of special cases like methane.
    
        Parameters
        ----------
        substance_dlci : Substance
            The substance from the dynamic life cycle inventory (DLCI).
        substance_cc : Substance
            The substance from the climate change (CC) impact assessment model.
    
        Returns
        -------
        tuple
            Returns the atmospheric concentration (F), dynamic temperature change (FTD),
            dynamic radiative forcing (RFD), and CO2 value from methane oxidation (value_co2).
        """
    
        # Unpack substance parameters for easier access
        factor, acc, a0, a1, a2, a3, tau1, tau2, tau3, correction = (
            substance_cc.factor,
            substance_cc.acc,
            substance_cc.a0,
            substance_cc.a1,
            substance_cc.a2,
            substance_cc.a3,
            substance_cc.tau1,
            substance_cc.tau2,
            substance_cc.tau3,
            substance_cc.correction
        )
    
       
        time_points = np.array(self.time_cc_calculation)
        
        # Impulse Response Functions (IRF) for radiative forcing of CO2 only (Joos 2013)  Data from IPCC AR5 table 8.SM.10
        accCO2, a0CO2, a1CO2, a2CO2, a3CO2, tau1CO2, tau2CO2, tau3CO2 = 28.96/44.01*1E9/5.15E18*0.0000137, 0.2173, 0.224, 0.2824, 0.2763, 394.4, 36.54, 4.304
        IRFCO2 = a0CO2 + a1CO2 * np.exp(-time_points / tau1CO2) + a2CO2 * np.exp(-time_points / tau2CO2) + a3CO2 * np.exp(-time_points / tau3CO2)
              
        # Impulse Response Functions (IRF) for radiative forcing for any substance
        IRF = a0 + a1 * np.exp(-time_points / tau1) + a2 * np.exp(-time_points / tau2) + a3 * np.exp(-time_points / tau3)
                    
        # Temperature response function (IRFT)  Data from IPCC AR5 table 8.SM.9
        f1, f2, d1, d2 = 0.631, 0.429, 8.4, 409.5  # Constants for temperature change impulse response
        IRFT = (f1 / d1) * np.exp(-time_points / d1) + (f2 / d2) * np.exp(-time_points / d2)
    
        # Carbon-climate feedback model (impulse response function IRFCF) data from Gasser 2017
        gamma, alpha1, alpha2, alpha3, to1, to2, to3 = 3.015E12, 0.6368, 0.3322, 0.0310, 2.376, 30.14, 490.1  #gamma in kg/(yr K), tau in yr

        
        # DIRAC function
        def DIRAC(t):
            return 1 if t == 0 else 0

        IRFCF = gamma * np.array([DIRAC(t) for t in time_points]) - gamma * ((alpha1/to1) * np.exp(-time_points / to1) + (alpha2/to2) * np.exp(-time_points / to2) + (alpha3/to3) * np.exp(-time_points / to3))
 
        
        # Dynamic concentration (F), radiative forcing (RFD), and temperature change (FTD) for each substance
        E = np.array([self.emit(t, substance_dlci) for t in time_points])
        F = np.convolve(E, IRF)[:len(time_points)] * self.step_calculation
        
        RFDdirect = F * factor * correction * acc
        
        RFDindirect, CP1, CP2, CP3 = 0.0, 0.0, 0.0, 0.0
        
        if substance_dlci.name not in ["Carbon dioxide", "Carbon dioxide, biogenic", "Carbon dioxide, in air", "Carbon dioxide, fossil", "Carbon dioxide, land transformation"]:
            CP1 = np.convolve(RFDdirect, IRFT)[:len(time_points)]* self.step_calculation
            CP2 = np.convolve(CP1, IRFCF)[:len(time_points)]* self.step_calculation
            CP3 = np.convolve(CP2, IRFCO2)[:len(time_points)]* self.step_calculation
            RFDindirect = CP3*accCO2
        
        RFD = RFDdirect + RFDindirect
        
        FTD = np.convolve(RFD, IRFT)[:len(time_points)] * self.step_calculation
    
        # CO2-flux formation from methane oxidation 1 kg CH4 gives alphaCO2 kg CO2    alphaCO2 = 0.75 from , Data IRF from IPCC AR5 ch8; 8.SM.11.3.2; Bucher 2009  changed by Ligia 07/2024
        alphaCO2 = 0.75   # reference 
        value_co2 = np.zeros(len(time_points))
        if substance_dlci.name in ["Methane", "Methane, non-fossil", "Methane, fossil", "Methane, from soil or biomass stock"]:
            IRF2 = a0 + a1 * np.exp(-time_points / tau1) + a2 * np.exp(-time_points / tau2) + a3 * np.exp(-time_points / tau3)
         #  methane = np.convolve(E, IRF2)[:len(time_points_CH4)] * self.step_calculation    # remaining flux of methane
            value_co2 = alphaCO2 * (44.01 / 16.04) * np.convolve(E, (1-IRF2))[:len(time_points)] * self.step_calculation 
         # verified: fraction of CO2 formed (balance CO2 + CH4 = 1)  value_co2 = np.convolve(E, (1-IRF2))[:len(time_points_CH4)] * self.step_calculation # Print methane handling result         
         # additional RF and T to CH4 results, from the neo-formed CO2
            RFD = RFD + accCO2 * np.convolve(value_co2, IRFCO2)[:len(time_points)] * self.step_calculation   
            FTD = FTD + np.convolve(RFD, IRFT)[:len(time_points)] * self.step_calculation    
          
     
        return F, FTD, RFD

   
    def methane_oxidation(self):
        """
        Performs the convolution calculation and the impacts calculation for a given
        substance
        """
        substances_to_remove = []
        for substance in self.substances_dlci[::-1]:
            if (substance.name in ['Methane', 'Methane, non-fossil', 'Methane, fossil', 'Methane, from soil or biomass stock']
                and substance.compartment == 'Air'
                and sum(substance.emission) != 0.0):
                
                substance_cc = next((x for x in self.cc_substances
                                     if x.name == substance.name
                                     and x.compartment == substance.compartment), None)
                
                if not substance_cc:
                    continue

                results_convolution, dyn_temperature, dyn_radiative_forcing = self.convolution_cc(substance, substance_cc)

                #self.write_results(substance.name, substance.compartment, substance.subcompartment, results_convolution, 'IRF', self.path['cc'])
                self.write_results(substance.name, substance.compartment, substance.subcompartment, dyn_temperature, 'temperature', self.path['cc'])
                self.write_results(substance.name, substance.compartment, substance.subcompartment, dyn_radiative_forcing, 'radiative', self.path['cc'])

                int_dyn_r = self.impacts_integral(dyn_radiative_forcing, self.time_cc_calculation)
                value_dynt = dyn_temperature[-1]
                temp_is_nan = np.isnan(np.sum(dyn_temperature))

                if not self.global_gtp_result:
                    self.global_gtp_result = np.zeros_like(dyn_temperature)
                if np.sum(dyn_temperature) != 0.0 and not temp_is_nan:
                    self.global_gtp_result += dyn_temperature

                if not self.global_gwp_result:
                    self.global_gwp_result = np.zeros_like(dyn_radiative_forcing)
                if np.sum(dyn_radiative_forcing) != 0.0 and not temp_is_nan:
                    self.global_gwp_result += dyn_radiative_forcing

                int_mass = np.sum(substance.emission) * self.step_dlci
                self.graphs_cc(dyn_temperature, substance.name, substance.subcompartment)
                self.report.append([substance.name, substance.subcompartment, value_dynt, int_dyn_r, int_mass])

                # mark methane for removal
                substances_to_remove.append(substance)

        # Remove methane substances
        for substance in substances_to_remove:
            self.substances_dlci.remove(substance)
            
    def impacts_integral (self, radiative, time):
        """
        calculate the value of the impacts integral

        parameters
        ----------
        radiative (list): vector of radiative forcing

        returns
        -------
        integral_radiative (float): integral of the dynamic radiative forcing

        """
        step = ((time[-1])-time[0])/(len(time)-1)
        integral_radiative = sum(radiative)*step

        return integral_radiative


    def emit(self, tt, substance):
        """
        Interpolation for the dlci inventory vector

        Calculates the dlci inventory value that corresponds to a specific time
        by interpolation

        Parameters
        ----------
        tt (float): discrete time for the interpolation
        substance (CC_substances object): substance which presented in the
        dynamic LCI results (emission vector can be found as an instance of the
        substance object)

        Returns
        -------
        az (float): dynamic inventory value after interpolation
        """
        if tt >= self.time_dlci_norm[-1]:
            return 0.0
        if len(self.time_dlci_norm) != len(substance.emission):
            raise ValueError(f"Length mismatch: time_dlci_norm ({len(self.time_dlci_norm)}) and substance.emission ({len(substance.emission)})")

        # Perform linear interpolation using NumPy
        az = np.interp(tt, self.time_dlci_norm, substance.emission)
        return az
    
    
    def cumulated_calculation(self):
            global_gwp_result = np.array(self.global_gwp_result)
            global_gtp_result = np.array(self.global_gtp_result)
            
            # Initialize cumulated arrays
            self.cumulated_rf = np.zeros_like(global_gwp_result)
            self.cumulated_temp = np.zeros_like(global_gtp_result)
            
            # Perform cumulative calculations
            self.cumulated_rf[0] = global_gwp_result[0] * self.step_calculation
            self.cumulated_rf[1:] = np.cumsum(global_gwp_result[1:] * self.step_calculation) + self.cumulated_rf[0]
    
            self.cumulated_temp[0] = global_gtp_result[0] * self.step_calculation
            self.cumulated_temp[1:] = np.cumsum(global_gtp_result[1:] * self.step_calculation) + self.cumulated_temp[0]
    # --------------------------------------------------------------------------
    #
    #  The following methods are only used for the support of this module and
    #  are not part of the main algorithm
    #
    # --------------------------------------------------------------------------

    def _add_substance(self, substances_dict):
        """
        Add a substance in the Climate_change class vector instance substances
        """
        self.cc_substances.append(Cc_substances(substances_dict))

    def import_cc_files(self):
        """
        Import the climate change files and create new object of the class
        Cc_substances
        """
        currentdir = os.path.dirname(os.path.dirname(os.path.abspath("__file__")))
        # import table of substances (link between DLCI and USEtox)
        opendir = open(currentdir + "\\data\\cc_substances_updated.csv", "r")
        reader = csv.reader(opendir, delimiter=',')
        sub_cc = list(reader)
        opendir.close()
        sub_cc = sub_cc[1:]
        opendir = open(currentdir + "\\data\\cc_sp_subs_updated.csv", "r")
        reader = csv.reader(opendir, delimiter=',')
        sp_subs_cc = list(reader)
        opendir.close()
        sp_subs_cc = sp_subs_cc[1:]

        for i in sp_subs_cc:
            x = [item for item in sub_cc if int(item[0]) == int(i[3])] #i[3]: id_cc_subs

            try:
                x = x[0]
            except IndexError:
                continue
            try:
                # convert Wm-2ppvb-1 to Wm-2kg-1
                ax = (28.96/float(i[7]))*(1E9/5.15E18)*float(x[3])
            except ZeroDivisionError:
                ax = 0
            # Acc*(MA/Mx)*(10E9/TM)
            #MA is the substance molecular weight or air (28.96 kg kmol-1)
            #Mx is the molecular weight of the substance x
            #TM is the total mass of the atmosphere (about 5.15E18 kg)
            # from Shine et al. 2005
            substances_dict = {'name': i[1],
                               'compartment': i[2],
                               'factor':float(i[5]),
                               'c_gtp100':float(x[16]),
                               'c_gwp100':float(x[13]),
                               'acc': ax,
                               'a0':float(x[4]),
                               'a1':float(x[5]),
                               'a2':float(x[6]),
                               'a3':float(x[7]),
                               'tau1':float(x[8]),
                               'tau2':float(x[9]),
                               'tau3':float(x[10]),
                               'correction':float(x[11])
                              }
            self._add_substance(substances_dict)

    def graphs_cc (self, dyn_temperature, substance, compartment):
        """
        Create the graphics for each susbtance
        """
        compartment = compartment.replace('.','')
        create_graphic('GTMC ' + str(substance) + '(' + str(compartment) +')',
                        # 'Total mass variation [reaction only system] (Removal mass)'+ str(name_file),
                        'Global Mean Temperature Change for \n' + str(substance) + ' in ' + str(compartment),
                        self.time_cc ,
                        dyn_temperature,
                        ' ',
                        'Time (years)',
                        'GMTC (K)' ,
                        self.path['g_cc'] + 'substances',
                        log_choice = False,
                        max_value_y1 = 1.1*max(dyn_temperature),
                        min_value_y = min(dyn_temperature),
                        with_legend = False
                        )


    def final_graphs_cc(self, radiative_forcing, cumulated_radiative_forcing, dyn_temperature, cumulated_temp):
        """
        Creates the final graphics for the dynamic cc method
        """
        def create_graph(title, ylabel, ydata, filename_suffix):
            max_value_y1 = 1.1 * np.max(ydata)
            min_value_y = np.min(ydata)
            create_graphic(title,
                           title,
                           self.time_cc,
                           ydata,
                           ' ',
                           'Time (years)',
                           ylabel,
                           self.path['g_cc'],
                           log_choice=False,
                           max_value_y1=max_value_y1,
                           min_value_y=min_value_y,
                           with_legend=False)

        create_graph('Global Mean Temperature Change', 'GMTC (K)', dyn_temperature, 'Global_Mean_Temperature_Change')
        create_graph('Radiative Forcing', 'RF (W.m-²)', radiative_forcing, 'Radiative_Forcing')
        create_graph('Cumulative Radiative Forcing', 'iRF (W.m-².year)', cumulated_radiative_forcing, 'Cumulative_Radiative_Forcing')
        create_graph('Cumulative Temperature', 'Cumulated_temp(K)', cumulated_temp, 'Cumulative_Temperature')
        
    def write_results(self,substance,compartment,subcompartment,results,category,file_path):
        """
        Write the results for the substances
        """
        file_results = file_path +'results_' + category + '.csv'
        file_temp = file_path + 'temp.csv'
        if not os.path.isfile(file_results):
            with open(file_results,'w') as f:
                writer = csv.writer(f,delimiter=',',lineterminator = '\n')
                writer.writerow(['time(years)'])
                for row in range(len(self.time_cc)):
                     writer.writerow([self.time_cc[row]])
        with open (file_results,'r') as csvinput:
            with open (file_temp,'w') as csvoutput:
                reader = csv.reader(csvinput)
                writer = csv.writer(csvoutput,delimiter=',',lineterminator = '\n')
                row0 = next(reader)
                row0.append(substance+'('+compartment+'/'+subcompartment+')')
                writer.writerow(row0)
                counter = 0
                for row in reader:
                    row.append(results[counter])
                    writer.writerow(row)
                    counter += 1
        os.remove(file_results)
        os.rename(file_temp,file_results)

    def write_final_results(self,file_path):
        """
        Write the final results for the dynamci .... method
        """
        file_results = file_path +'results_total.csv'
        with open(file_results,'w') as f:
            writer = csv.writer(f,delimiter=',',lineterminator = '\n')
            writer.writerow(['time(years)','RF, W/m²', 'cum. RF, W.y/m²', 'temp, K','cum. temp, K.y']) # add cumulated temp
            results = list(zip(self.time_cc, self.global_gwp_result, self.cumulated_rf,self.global_gtp_result,self.cumulated_temp))
            for row in range(len(results)):
                 writer.writerow(results[row])
